//------------------------------------------------------------------------------
// File: cmn_slgameobjlist.cs
// Contains algorithms to be performed on game object lists
// Author: Matthew Rudge
//------------------------------------------------------------------------------

////////////////////////////////////////////////////////////////////////////////
//! Checks object list to see if it contains the named object
//! \param %objName - Name of object to find
//! \retval bool True if the object was found, false otherwise
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::containsName(%this, %objName)
{
   // Loop through list and check for object by name
   for(%i = 0; %i < %this.getSize(); %i++) {
      %obj = %this.getID(%i);
      if(%obj.name $= %objName) {
         return true;
      }
   }
   
   // Default is false
   return false;
}

////////////////////////////////////////////////////////////////////////////////
//! Checks object list to see if it contains the internally named object
//! \param %intName - Internal name of object
//! \retval bool True if the object was found, false otherwise
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::containsInternalName(%this, %intName)
{
   // Loop through list and check for object by name
   for(%i = 0; %i < %this.getSize(); %i++) {
      %obj = %this.getID(%i);
      %internalName = %obj.getInternalName();
      if(%internalName $= %intName) {
         return true;
      }
   }
   
   // Default is false
   return false;
}

////////////////////////////////////////////////////////////////////////////////
//! Checks object list to see if it contains the object type
//! \param %objType - Name of object type to find
//! \retval bool True if the object was found, false otherwise
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::containsType(%this, %objType)
{
   // Loop through list and check for object type
   for(%i = 0; %i < %this.getSize(); %i++) {
      %obj = %this.getID(%i);
      if (isObject(%obj) == false)
      {
         continue;
      }
      if(%obj.isOfType(%objType)) {
         return true;
      }
   }
   
   // Default is false
   return false;
}

////////////////////////////////////////////////////////////////////////////////
//! Checks object list to see if it contains the object
//! \param %objType - Name of object type to find
//! \param %objTeam - Team of object to find
//! \retval bool True if the object was found, false otherwise
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::containsObject(%this, %objType, %objTeam)
{
   // For each object do
   %size = %this.getSize();
   for(%i = 0; %i < %size; %i++) {
      %obj = %this.getID(%i);
      if (isObject(%obj) == false)
      {
         continue;
      }
      
      // If the object is of the type desired then
      if(%obj.isOfType(%objType)) {
         if(%obj.isOnTeam(%objTeam)) {
            return true;
         }
      }
   }
   
   // Default is false
   return false;
}

////////////////////////////////////////////////////////////////////////////////
//! Checks object list to see if it contains a team object
//! \param %objTeam - Team of object to find
//! \retval bool True if the object was found, false otherwise
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::containsTeam(%this, %objTeam)
{
    // For each object do
   %size = %this.getSize();
   for(%i = 0; %i < %size; %i++) {
      %obj = %this.getID(%i);
      if (isObject(%obj) == false)
      {
         continue;
      }
      
      // If this object is on the team then
      if(%obj.isOnTeam(%objTeam)) {
         return true;
      }
   }
   
   // Default is false
   return false;
}

////////////////////////////////////////////////////////////////////////////////
//! Checks to see if all game objects are on the selected team
//! \param %team Team to query
//! \retval bool True if all are on the team, false otherwise
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::isOnTeam(%this, %team)
{
   // Loop through list and check all individuals for team
   for(%i = 0; %i < %this.getSize(); %i++) {
      if(%this.getID(%i).getTeam() != %team) {
         return false;
      }
   }
   
   // All objects on same team
   return true;
}

////////////////////////////////////////////////////////////////////////////////
// Checks to see if all game objects are of a certain type
//! \param %type Type of game object ($CID_PROP, $CID_CHARACTER, $CID_BUILDING)
//! \retval bool True if all objects are of type, false otherwise
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::isOfObjectType(%this, %type)
{
   // Loop through list and check all individuals for type
   for(%i = 0; %i < %this.getSize(); %i++) {
      %obj = %this.getID(%i);
      %cmp = slgQueryInterface(%obj, %type);
      if(!isObject(%cmp)) {
         return false;
      }
   }
   
   // All are of type
   return true;
}

////////////////////////////////////////////////////////////////////////////////
//! Returns a list of objects that match type
//! \param %objType Type of objects
//! \param %objList List of objects that match type (out)
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::getTypeList(%this, %objType, %objList)
{
   if(!isObject(%objList)) {
      return;
   }
   
   %objList.clear();
   %size = %this.getSize();
   for(%i = 0; %i < %size; %i++) {
      %obj = %this.getID(%i);
      if(%obj.isOfType(%objType)) {
         %objList.addObject(%obj);
      }
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the object with the given name if found
//! \param %objName Object name
//! \retval int Id of object with object name or zero
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::getObjectByName(%this, %objName)
{
   %size = %this.getSize();
   for(%i = 0; %i < %size; %i++) {
      %obj = %this.getID(%i);
      if(%obj.name $= %objName) {
         return %obj;
      }
   }
   return 0;
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the object with the given internal name if found
//! \param %intName Internal name
//! \retval int Id of object with internal name or zero
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::getObjectByInternalName(%this, %intName)
{
   %size = %this.getSize();
   for(%i = 0; %i < %size; %i++) {
      %obj = %this.getID(%i);
      if(%obj.getInternalName() $= %intName) {
         return %obj;
      }
   }
   return 0;
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the first object of type if found
//! \param %type Type of object
//! \retval int Id of first type object or zero
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::getObjectByType(%this, %type)
{
   %size = %this.getSize();
   for(%i = 0; %i < %size; %i++) {
      %obj = %this.getID(%i);
      if(%obj.isOfType(%type)) {
         return %obj;
      }
   }
   return 0;
}

////////////////////////////////////////////////////////////////////////////////
//! Returns a list of objects that match team
//! \param %objTeam Team of objects
//! \param %objList List of objects that match team (out)
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::getTeamList(%this, %objTeam, %objList)
{
   if(!isObject(%objList)) {
      return;
   }
   
   %objList.clear();
   %size = %this.getSize();
   for(%i = 0; %i < %size; %i++) {
      %obj = %this.getID(%i);
      if(%obj.isOnTeam(%objTeam)) {
         %objList.addObject(%obj);
      }
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Returns a random object in the game object list
//! \retval integer Valid object or 0 if list is empty
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::getRandomObject(%this)
{
   %size = %this.getSize();
   
   // Empty list
   if(%size <= 0) {
      return 0;
   }
   
   // Get random object from list
   return (%this.getID(getRandom(0, %size - 1)));
}

////////////////////////////////////////////////////////////////////////////////
//! Appends a sim group to this one
//! \param %group SLGameObjList group to append
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::appendGroup(%this, %group)
{
   if(!isObject(%group)) {
      return;
   }
   
   for(%i = 0; %i < %group.getSize(); %i++) {
      %this.addObject(%group.getID(%i));
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Removes all sim group objects from this one
//! \param %group SLGameObjList group of objects to remove
////////////////////////////////////////////////////////////////////////////////
function SLGameObjList::removeGroup(%this, %group)
{
   if(!isObject(%group)) {
      return;
   }
   
   for(%i = 0; %i < %group.getSize(); %i++) {
      %this.removeObject(%group.getID(%i));
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Fills out a game object list of buildings that have tenant space
//! \param %shelters List of buildings with tenat space returned by this method
////////////////////////////////////////////////////////////////////////////////
function slgGetTenantAvailability(%tenantBldgs)
{
   if(!isObject(%tenantBldgs)) {
      return;
   }
   
   // Clear list
   %tenantBldgs.clear();
   
   // Get shelters
   %bldgs = slgGetServerBuildingList();
   for(%i = 0; %i < %bldgs.getSize(); %i++) {
      slgAddTenantBldg(%bldgs.getID(%i), %tenantBldgs);
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Adds a building to the tenant list if it has tenant availability
//! \param %idBldg Id of building
//! \param %tenantBldgs Game object list to add building to if it has tenant 
//! space
////////////////////////////////////////////////////////////////////////////////
function slgAddTenantBldg(%idBldg, %tenantBldgs)
{
   %ccmpbldg = slgQueryInterface(%idBldg, $CID_BUILDING);
   if(!isObject(%ccmpbldg)) {
      return;
   }
   
   if(%ccmpbldg.canAddTenant()) {
      %tenantBldgs.addObject(%idBldg);
   }
}

// End cmn_slgameobjlist.cs